/*
 * Decompiled with CFR 0.152.
 */
package com.initech.provider.crypto.cipher;

import com.initech.cryptox.BadPaddingException;
import com.initech.cryptox.Cipher;
import com.initech.cryptox.IllegalBlockSizeException;
import com.initech.cryptox.KSXRuntimeException;
import com.initech.cryptox.NoSuchPaddingException;
import com.initech.cryptox.SecretKey;
import com.initech.cryptox.SecretKeyFactory;
import com.initech.cryptox.spec.IvParameterSpec;
import com.initech.cryptox.spec.SecretKeySpec;
import com.initech.cryptox.util.Hex;
import com.initech.provider.crypto.InitechProvider;
import com.initech.provider.crypto.SelfTestCase;
import com.initech.provider.crypto.cipher.Aria;
import com.initech.provider.crypto.cipher.AriaKey;
import com.initech.provider.crypto.cipher.RSA;
import com.initech.provider.crypto.cipher.Seed;
import com.initech.provider.crypto.cipher.SeedKey;
import com.initech.provider.crypto.rsa.RSAPrivateKeyImpl;
import com.initech.provider.crypto.rsa.RSAPublicKeyImpl;
import com.initech.provider.crypto.spec.RSAOAEPParameterSpec;
import java.math.BigInteger;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.RSAPrivateKeySpec;
import java.security.spec.RSAPublicKeySpec;
import java.util.Arrays;

public final class CipherSelfTestCase
implements SelfTestCase {
    void testRSAES() {
        byte[] input = Hex.parseHexaString("a54ee183a0c004cf49f640d62f8f59bff4484e778dd5d5fe92684e76d89034413dfa7f835178efa7b78810548848e234cd0a71864c3db9601c99a2ce3255");
        String modulusStr = "a158d0b6a1a1b401b156c8fa50acb2f4ae8c9ca2c73f7a729d0eb54aa2fcb6be4f7ee6946904eba21a10e522fe6ff919fe2fd38f5580f60abb62b02c2ed4a117a74bdd101def4b1dea200c49ba0738a30cb9b13d81caad9f5b42238d0f61389c07ba7de10f15cf243fab5feb39e44484e986e5ff6203d757399b2ee11f11c932ee708f28d151a6df80c3bfca5a6129bd0866baf1a0fca02914ca844da9b876282508e33714c2a3841c92b77c33ab4141ef3c3770e36ddf6a9e0a18e529d097a7f39d64ae0b02fd496d9ad3705bf078dbd77ad1b5923995ffd9ff7f951d04f07ea3fb687862400985701a6f0ddc544d17df6a0fe3e1108f425ab4286502f9ddb3";
        String privateExponentStr = "01e5ec96a972208f008a17d5320bc61330d680a53d98d694f09a12b1ed16f2036d25fe0a72edd6e3d250ae9169d2e16857e03229f6848ce07571e4c6b04a5a16d0260954be356979d19e4ce6b5248e9b04744ecc7d7167f77ef85938b89c9930e4ba61eebae1b3cd265e9ac346ac13ce487d08c9a75a8a37017b7ce2b31ddf7310131f188d82e94bd2b6353d3202e9601d5f0f58376d5650e2078c7c44ab335bf67a064feddfad315ddec098c8a283ca744294ce40921ceb84ecc7b19779de8bb9709dc66a234f5266cbb98b3fe2bb540108e7ee7746d45b28b2a08a6a7b37ed59cba13caf03f88a044b684344e26cbbc38668b163c31e94151c32df4fe2db61";
        String publicExponentStr = "10001";
        String primePStr = "f553f7b4e860b649ced3179d51c09c83c83f25e3bd62899211cbbefbffbbe76f21837c895cdd2c42bc56f77c34d0c6beae9bcc8c230821e0bf40eda1338718291747bfacb3217137491771f78f9d563a9323af7e6b5cefb512c5cf3fbc304609c530823cfcd3d934ff93c0d9726da307d6a82020d143687d4a1f49c0ee4f1491";
        String primeQStr = "a85d9d349f13341388c1db2cf357600ed583518f208343577a628af0fd8107dc989c0cb64a1f8cd2c9730b86b9b0d52b3ef466a1e323d59139ff15e4477ba0d586e35a496c8f0caa2d1aad2f468533a6ca834e40b63418d934344c0b53abb093827e302c09e532a47ffdb24bf07a371d8fd6c85105abd5af95f77d844912a003";
        try {
            BigInteger modulus = new BigInteger(modulusStr, 16);
            BigInteger privateExponent = new BigInteger(privateExponentStr, 16);
            BigInteger publicExponent = new BigInteger(publicExponentStr, 16);
            BigInteger primeP = new BigInteger(primePStr, 16);
            BigInteger primeQ = new BigInteger(primeQStr, 16);
            RSAPublicKeyImpl rsapublickey = new RSAPublicKeyImpl(modulus, publicExponent);
            RSAPrivateKeyImpl rsaprivatecrtkey = new RSAPrivateKeyImpl(modulus, privateExponent);
            RSA rsa = new RSA();
            rsa._engineInit(1, rsapublickey, new SecureRandom());
            rsa._engineSetPadding("OAEPPadding");
            byte[] ciphertext = rsa._engineDoFinal(input, 0, input.length);
            rsa._engineInit(2, rsaprivatecrtkey, new SecureRandom());
            rsa._engineSetPadding("OAEPPadding");
            byte[] output = rsa._engineDoFinal(ciphertext, 0, ciphertext.length);
            if (!Arrays.equals(input, output)) {
                throw new KSXRuntimeException("RSAOAEP SelfTest Failed");
            }
        }
        catch (KSXRuntimeException e) {
            throw e;
        }
        catch (Exception e) {
            throw new KSXRuntimeException("RSAOAEP SelfTest Error", e);
        }
    }

    void testSEED() {
        String rawkey = "00000000000000000000000000000000";
        String plaintext = "000102030405060708090A0B0C0D0E0F";
        String ciphertext = "5EBAC6E0054E166819AFF1CC6D346CDB";
        try {
            SeedKey key = new SeedKey(Hex.parseHexaString(rawkey));
            Seed seed = new Seed();
            byte[] output = new byte[16];
            seed.setKey(key);
            seed.encryptBlock(Hex.parseHexaString(plaintext), 0, 16, output, 0);
            if (!Arrays.equals(Hex.parseHexaString(ciphertext), output)) {
                throw new KSXRuntimeException("SEED SelfTest Failed");
            }
        }
        catch (Exception e) {
            throw new KSXRuntimeException("SEED SelfTest Error", e);
        }
    }

    void testARIA128() throws KSXRuntimeException {
        String rawkey = "00000000000000000000000000000000";
        String plaintext = "80000000000000000000000000000000";
        String ciphertext = "92E51E737DABB6BFD0EABC8D32224F77";
        try {
            AriaKey key = new AriaKey(Hex.parseHexaString(rawkey));
            Aria aria = new Aria();
            byte[] output = new byte[16];
            aria.setKey(key);
            aria.encryptBlock(Hex.parseHexaString(plaintext), 0, 16, output, 0);
            if (!Arrays.equals(Hex.parseHexaString(ciphertext), output)) {
                throw new KSXRuntimeException("ARIA SelfTest Failed");
            }
        }
        catch (KSXRuntimeException e) {
            throw e;
        }
        catch (Exception e) {
            throw new KSXRuntimeException("ARIA SelfTest Error", e);
        }
    }

    void testARIA192() throws KSXRuntimeException {
        String rawkey = "000000000000000000000000000000000000000000000000";
        String plaintext = "80000000000000000000000000000000";
        String ciphertext = "AE56BF59874ED4DBD7B905878894ADEE";
        try {
            AriaKey key = new AriaKey(Hex.parseHexaString(rawkey));
            Aria aria = new Aria();
            byte[] output = new byte[16];
            aria.setKey(key);
            aria.encryptBlock(Hex.parseHexaString(plaintext), 0, 16, output, 0);
            if (!Arrays.equals(Hex.parseHexaString(ciphertext), output)) {
                throw new KSXRuntimeException("ARIA SelfTest Failed");
            }
        }
        catch (KSXRuntimeException e) {
            throw e;
        }
        catch (Exception e) {
            throw new KSXRuntimeException("ARIA SelfTest Error", e);
        }
    }

    void testARIA256() throws KSXRuntimeException {
        String rawkey = "0000000000000000000000000000000000000000000000000000000000000000";
        String plaintext = "80000000000000000000000000000000";
        String ciphertext = "209ACF63CE6DFB56B1F9821DDBFE86E3";
        try {
            AriaKey key = new AriaKey(Hex.parseHexaString(rawkey));
            Aria aria = new Aria();
            byte[] output = new byte[16];
            aria.setKey(key);
            aria.encryptBlock(Hex.parseHexaString(plaintext), 0, 16, output, 0);
            if (!Arrays.equals(Hex.parseHexaString(ciphertext), output)) {
                throw new KSXRuntimeException("ARIA SelfTest Failed");
            }
        }
        catch (KSXRuntimeException e) {
            throw e;
        }
        catch (Exception e) {
            throw new KSXRuntimeException("ARIA SelfTest Error", e);
        }
    }

    void testARIA128ECB() throws KSXRuntimeException {
        byte[] key = Hex.hexToByteArray("9F118FC486D031868D168D24D8058FF1");
        byte[] iv = Hex.hexToByteArray("3FA76B5850DCCC7197972ED21D4A88A8");
        byte[] msg = Hex.hexToByteArray("2E9F2F065F69F77CD0DA9BECAFF65F99FD47F38195D39BED6A4B8D4610859A2A7E2C11BF12B7E147D597FDF501B505FFC01B22F537F13251C3AC254F8282A6FF");
        byte[] cipher = Hex.hexToByteArray("96d2c8b8ecfd71f0c728a14ad43539689ac18e8f36ab1c53e993763acde8931ffde30269c0fc9c9508a940d81227ac947654740190ef6b13d7a5c7e72eaa9862");
        byte[] encData = CipherSelfTestCase.symmetric(msg, "ARIA/ECB/NoPadding", "ARIA", key, iv, 1);
        byte[] decData = CipherSelfTestCase.symmetric(encData, "ARIA/ECB/NoPadding", "ARIA", key, iv, 2);
        if (!Arrays.equals(cipher, encData) || !Arrays.equals(msg, decData)) {
            throw new KSXRuntimeException("testARIA128ECB SelfTest Failed");
        }
    }

    void testARIA128CBC() throws KSXRuntimeException {
        byte[] key = Hex.hexToByteArray("1621FE98A5A0360F24B67CE8D62AC94D");
        byte[] iv = Hex.hexToByteArray("0F8AC05D3E20D8C62A22BD967C29F44E");
        byte[] msg = Hex.hexToByteArray("7EE4BE38220E40FB469DE06D363DF2A873FC9460B347D67428151FC20562CC403FDF194990E84534C2B6F12B2AD6CA3A73D967A79D4E748244ED117C261575FDE079D770F8168EE2206877CBB9DD972B");
        byte[] cipher = Hex.hexToByteArray("c9f5e10600732a7ebc8708edd12c0aa3d478dc6fa98415dc6e1729907471e1ab7beee26daa4428ea28ebe927d00029901e65caa885f40c53bd3fd91a5ae7defb781d853f94ed802d436d62aa124dc859");
        byte[] encData = CipherSelfTestCase.symmetric(msg, "ARIA/CBC/NoPadding", "ARIA", key, iv, 1);
        byte[] decData = CipherSelfTestCase.symmetric(encData, "ARIA/CBC/NoPadding", "ARIA", key, iv, 2);
        if (!Arrays.equals(cipher, encData) || !Arrays.equals(msg, decData)) {
            throw new KSXRuntimeException("testARIA128CBC SelfTest Failed");
        }
    }

    void testARIA128CFB8() throws KSXRuntimeException {
        byte[] key = Hex.hexToByteArray("CDE6EC1462D93EE1F47A9668EF165D80");
        byte[] iv = Hex.hexToByteArray("A151F9074019F0762FD851B87DEE503D");
        byte[] msg = Hex.hexToByteArray("AF5E304ABFC766936497FB0B947B32FB471A6B517F02473B85E20DF575FB4CE0FCD1C2D361267DB5C228404CA1EA2850");
        byte[] cipher = Hex.hexToByteArray("ba0bcf5550628e136c68962aae618354350f8b894a5d80a3d74669adf2d70d7eeb407278a8402f121348f615213a730b");
        byte[] encData = CipherSelfTestCase.symmetric(msg, "ARIA/CFB8/NoPadding", "ARIA", key, iv, 1);
        byte[] decData = CipherSelfTestCase.symmetric(encData, "ARIA/CFB8/NoPadding", "ARIA", key, iv, 2);
        if (!Arrays.equals(cipher, encData) || !Arrays.equals(msg, decData)) {
            throw new KSXRuntimeException("testARIA128CFB SelfTest Failed");
        }
    }

    void testARIA128CFB128() throws KSXRuntimeException {
        byte[] key = Hex.hexToByteArray("CDE6EC1462D93EE1F47A9668EF165D80");
        byte[] iv = Hex.hexToByteArray("A151F9074019F0762FD851B87DEE503D");
        byte[] msg = Hex.hexToByteArray("AF5E304ABFC766936497FB0B947B32FB471A6B517F02473B85E20DF575FB4CE0FCD1C2D361267DB5C228404CA1EA2850");
        byte[] cipher = Hex.hexToByteArray("ba2a92ecffdc61b58ae83d894a7f7f798525d4d768eff2b1fbe3ff4965dc55d49a425b3a1958d47189e470b3f918e216");
        byte[] encData = CipherSelfTestCase.symmetric(msg, "ARIA/CFB128/NoPadding", "ARIA", key, iv, 1);
        byte[] decData = CipherSelfTestCase.symmetric(encData, "ARIA/CFB128/NoPadding", "ARIA", key, iv, 2);
        if (!Arrays.equals(cipher, encData) || !Arrays.equals(msg, decData)) {
            throw new KSXRuntimeException("testARIA128CFB SelfTest Failed");
        }
    }

    void testARIA128OFB() throws KSXRuntimeException {
        byte[] key = Hex.hexToByteArray("1621FE98A5A0360F24B67CE8D62AC94D");
        byte[] iv = Hex.hexToByteArray("0F8AC05D3E20D8C62A22BD967C29F44E");
        byte[] msg = Hex.hexToByteArray("7EE4BE38220E40FB469DE06D363DF2A873FC9460B347D67428151FC20562CC403FDF194990E84534C2B6F12B2AD6CA3A73D967A79D4E748244ED117C261575FDE079D770F8168EE2206877CBB9DD972B");
        byte[] cipher = Hex.hexToByteArray("276661DB9057C89FA6FB1C7193100EB59B2C9FF864541CCFBA02B1AD4C733487340B560B6907D84EDCD85EE7668404ECEA29BFC4624753C630E6E00FB4178BCB06E8E4BD8D16900D08517BA92FDA20BD");
        byte[] encData = CipherSelfTestCase.symmetric(msg, "ARIA/OFB/NoPadding", "ARIA", key, iv, 1);
        byte[] decData = CipherSelfTestCase.symmetric(encData, "ARIA/OFB/NoPadding", "ARIA", key, iv, 2);
        if (!Arrays.equals(cipher, encData) || !Arrays.equals(msg, decData)) {
            throw new KSXRuntimeException("testARIA128OFB SelfTest Failed");
        }
    }

    void testARIA128CTR() throws KSXRuntimeException {
        byte[] key = Hex.hexToByteArray("978A01D9041CFA1807125C6CE62A38AC");
        byte[] iv = Hex.hexToByteArray("E919BF57617F632BE91A39F4EC3AA2A3");
        byte[] msg = Hex.hexToByteArray("67752A9FF09BB060F8EDC8394D895D75E2299BA6D30D0B3AA436004FCAD533C96C03AAA16AB2DE7F9FE41C8C631B2C815410310557A7D035D92394845B9691C52D9C48867949CBA08460220C31C6ECF8C736491AF335F8451149BD3AA76606C0");
        byte[] cipher = Hex.hexToByteArray("71555FCB9FA2CEEB8AA924EFCADA531DAC4F4B02299265B4A9359D33AA4DD0884BB5EC29147D43999990917A10008D42E85C230CE5137D9A17DF6E4919D0DA509CC800153A87EA7EEB5CED179CFE6300D718EC9721A2EB6858459D5FD3DD255B");
        byte[] encData = CipherSelfTestCase.symmetric(msg, "ARIA/CTR/NoPadding", "ARIA", key, iv, 1);
        byte[] decData = CipherSelfTestCase.symmetric(encData, "ARIA/CTR/NoPadding", "ARIA", key, iv, 2);
        if (!Arrays.equals(cipher, encData) || !Arrays.equals(msg, decData)) {
            throw new KSXRuntimeException("testARIA128CTR SelfTest Failed");
        }
    }

    void testARIA192ECB() throws KSXRuntimeException {
        byte[] key = Hex.hexToByteArray("1BB23C9DAF4BAFCDC261BFE5DD28615CAA74A841057C8125");
        byte[] iv = Hex.hexToByteArray("55DEA0341DE4BAA5CE06F555E0B09DAE");
        byte[] msg = Hex.hexToByteArray("204518CE4C631CE5DD691FDF24360F449834445C6F96B3966BDD642AF7A126FE728E81C7CBADEC704EE440BFBDC16D36");
        byte[] cipher = Hex.hexToByteArray("63f93c574fb2b3d3b110bf0f3183f3765b9b5ff57e5179e69c8ad720cc506e3846ef0cf4a0bffba32caf590282bc7c69");
        byte[] encData = CipherSelfTestCase.symmetric(msg, "ARIA/ECB/NoPadding", "ARIA", key, iv, 1);
        byte[] decData = CipherSelfTestCase.symmetric(encData, "ARIA/ECB/NoPadding", "ARIA", key, iv, 2);
        if (!Arrays.equals(cipher, encData) || !Arrays.equals(msg, decData)) {
            throw new KSXRuntimeException("testARIA192ECB SelfTest Failed");
        }
    }

    void testARIA192CBC() throws KSXRuntimeException {
        byte[] key = Hex.hexToByteArray("A3736B37E519746C9A4071661CA6B2845F44E117828C39C3");
        byte[] iv = Hex.hexToByteArray("A4F0F329F9A301C21DA3FF0EA8F767EF");
        byte[] msg = Hex.hexToByteArray("0235044F7948A909907022079A8A6BA888922064006635B2AFCDE729DFB7F2E64CC7052F330E7355AD1A2BDF3B2FEAE4A2D4AFB9F99170CD00F80BD0B3E27F1B");
        byte[] cipher = Hex.hexToByteArray("ec40d867136542f9554964083cd746d323b09a1c14916a9123d9afd94463f6cba2c42c0fefa8bb8b8082ca2881c7472f05f91a7e2056fc2f3af3ff0c48d2e364");
        byte[] encData = CipherSelfTestCase.symmetric(msg, "ARIA/CBC/NoPadding", "ARIA", key, iv, 1);
        byte[] decData = CipherSelfTestCase.symmetric(encData, "ARIA/CBC/NoPadding", "ARIA", key, iv, 2);
        if (!Arrays.equals(cipher, encData) || !Arrays.equals(msg, decData)) {
            throw new KSXRuntimeException("testARIA192CBC SelfTest Failed");
        }
    }

    void testARIA192CFB8() throws KSXRuntimeException {
        byte[] key = Hex.hexToByteArray("E05013824154DCECECCE35CADCA6FCD000D7C884D7BC22A2");
        byte[] iv = Hex.hexToByteArray("3D8BF2BE851F372BC7FAFB1C7B8A8709");
        byte[] msg = Hex.hexToByteArray("47ABD354166E402D6A1889B3F36EF75B6030D59675E424AE5ED490F1435AA2121C5B36CCC71230C8570F6BC1B07FF2E651AD738070C9F0D44CE9B84BBF4D96D012E7487A161A326B70C354F835767A0AB70AB2C39E5703653A3E5C7017EACB0B");
        byte[] cipher = Hex.hexToByteArray("386f54f7ca039aba79e159c9d34c2a84f8c4cf35987c3e5726c68d9bc506a2175184d9861044b552a574e507b06850d37c16ac30f2cf80e631bd0cc0ad3bc294b2b3857479cbd9339045c9bdba5749f4b57191dddcd1cbf9661d918cc06ed270");
        byte[] encData = CipherSelfTestCase.symmetric(msg, "ARIA/CFB8/NoPadding", "ARIA", key, iv, 1);
        byte[] decData = CipherSelfTestCase.symmetric(encData, "ARIA/CFB8/NoPadding", "ARIA", key, iv, 2);
        if (!Arrays.equals(cipher, encData) || !Arrays.equals(msg, decData)) {
            throw new KSXRuntimeException("testARIA192CFB8 SelfTest Failed");
        }
    }

    void testARIA192CFB128() throws KSXRuntimeException {
        byte[] key = Hex.hexToByteArray("D256EFA52C0FB0DC5D3C2D9EABDBF68E3A4D7CA92615C629");
        byte[] iv = Hex.hexToByteArray("CFDD65A974B9A98B04B01597307911E4");
        byte[] msg = Hex.hexToByteArray("C582DFFC3937D03B8380B878308B9FE8B4CDB93EFE4399174FFEA1964BDEBD4B519E515A08106FD93AAA4E79E642F76990164277DC0F01795B477BEBA5C81CD9A8976A003FF13B3007D425F46EC137750CC1E69C37A64A77D1938BDE65BE965672751435075F9C0790052731F090C5D4");
        byte[] cipher = Hex.hexToByteArray("ec06e0564bab07ae01a00e2a06e5fd5180c90919cff78d2613fd54ad5e6f57d3d403e32204771db014405ae2ee47a256320b7562e51825117ed1224f9ec277de3e502d6856369273d5cc0c072b2e227b5fd11f35636944b538c2e38565726425dfd51cc346df56708d186808e4685eb0");
        byte[] encData = CipherSelfTestCase.symmetric(msg, "ARIA/CFB128/NoPadding", "ARIA", key, iv, 1);
        byte[] decData = CipherSelfTestCase.symmetric(encData, "ARIA/CFB128/NoPadding", "ARIA", key, iv, 2);
        if (!Arrays.equals(cipher, encData) || !Arrays.equals(msg, decData)) {
            throw new KSXRuntimeException("testARIA192CFB128 SelfTest Failed");
        }
    }

    void testARIA192OFB() throws KSXRuntimeException {
        byte[] key = Hex.hexToByteArray("D0D48FF4368EDDDA59EAB9B6B44993895940364288E4FA27");
        byte[] iv = Hex.hexToByteArray("80453B7795380B4E835A24C802512069");
        byte[] msg = Hex.hexToByteArray("9A55EDBDB9F07B3B0301B76FE907BD597D9C491084406FAACD588D60C2767DEE3DDB0F399D3BB63418C0810453514D632F133A42E6305BB258D8B2049F865B2FE985077387B2AC3C41837B49ED48130D40B1F456E391362CC9E17BFCC10822F6475ABDB3A0DEC51B26548D86E0767521557F6094A2EA68E1CB7CD08F4F843909");
        byte[] cipher = Hex.hexToByteArray("17982f0d7e7a7a72e83a54195d364a205901c539af3ee939e0ea50203e2456f7d40e75657969c476e351888e736db8b15de6858aab123381d3957f21153649407900807d4a153d4327094981943964fa924312c286530c5db273c9facb665115872396fda8e879d6ecb56e2e3eff8f6a66897ff36018658e60b4c00807d36845");
        byte[] encData = CipherSelfTestCase.symmetric(msg, "ARIA/OFB/NoPadding", "ARIA", key, iv, 1);
        byte[] decData = CipherSelfTestCase.symmetric(encData, "ARIA/OFB/NoPadding", "ARIA", key, iv, 2);
        if (!Arrays.equals(cipher, encData) || !Arrays.equals(msg, decData)) {
            throw new KSXRuntimeException("testARIA192OFB SelfTest Failed");
        }
    }

    void testARIA192CTR() throws KSXRuntimeException {
        byte[] key = Hex.hexToByteArray("FF6219410F466A986E3A9F025363DC65198366434CC35999");
        byte[] iv = Hex.hexToByteArray("04B0990771830A30B8A50464FD72017D");
        byte[] msg = Hex.hexToByteArray("C13E99066F95B1A331C7EFAC08A4711C1C85BEAA518CFC351A2C646592EBF490C96663DA5B965AA14853E118FF161831BC0327C1142777DFCFFE238CF538A99A2BBCE6C640EE4128042D26CD59A0B6A28C33BC93E5DCE3F57C2327214FE18A62924808124623CCFF0D5FA4143DCBB335331C666AEA34A940CCA45A6CC76FFDB2A411B20594BF65F00CF14534D31E77B3");
        byte[] cipher = Hex.hexToByteArray("3FBE795FD9DDD6059C00FEB1C2CA945B3A89B4D267F695BDBF6A466268B3CFF5F733E3616C155EF82CA6B555FC722C5FDC7811C7BEDC36D1C48272C32F9BBABE99B20BCD4FD74883F0C9F2FF43709110507A222973C99836D3C2E1AB5B95D12F45CC68DE37CDA26FB0DE3C51682C73D95232B66BC4A541DF5FF43381EB15F559CCE6B016186B77AE3826A511B5D612BE");
        byte[] encData = CipherSelfTestCase.symmetric(msg, "ARIA/CTR/NoPadding", "ARIA", key, iv, 1);
        byte[] decData = CipherSelfTestCase.symmetric(encData, "ARIA/CTR/NoPadding", "ARIA", key, iv, 2);
        if (!Arrays.equals(cipher, encData) || !Arrays.equals(msg, decData)) {
            throw new KSXRuntimeException("testARIA192CTR SelfTest Failed");
        }
    }

    void testARIA256ECB() throws KSXRuntimeException {
        byte[] key = Hex.hexToByteArray("AA74A841057C812555DEA0341DE4BAA5CE06F555E0B09DAE204518CE4C631CE5");
        byte[] iv = Hex.hexToByteArray("DD691FDF24360F449834445C6F96B396");
        byte[] msg = Hex.hexToByteArray("6BDD642AF7A126FE728E81C7CBADEC704EE440BFBDC16D36A3736B37E519746C9A4071661CA6B2845F44E117828C39C3");
        byte[] cipher = Hex.hexToByteArray("21cd1ba4fec804428861dc0c4808f943a1ea5bca87f4c18d63994ae00fd406520d138829463b96a78d6627e065398cce");
        byte[] encData = CipherSelfTestCase.symmetric(msg, "ARIA/ECB/NoPadding", "ARIA", key, iv, 1);
        byte[] decData = CipherSelfTestCase.symmetric(encData, "ARIA/ECB/NoPadding", "ARIA", key, iv, 2);
        if (!Arrays.equals(cipher, encData) || !Arrays.equals(msg, decData)) {
            throw new KSXRuntimeException("testARIA256ECB SelfTest Failed");
        }
    }

    void testARIA256CBC() throws KSXRuntimeException {
        byte[] key = Hex.hexToByteArray("A4F0F329F9A301C21DA3FF0EA8F767EF0235044F7948A909907022079A8A6BA8");
        byte[] iv = Hex.hexToByteArray("88922064006635B2AFCDE729DFB7F2E6");
        byte[] msg = Hex.hexToByteArray("4CC7052F330E7355AD1A2BDF3B2FEAE4A2D4AFB9F99170CD00F80BD0B3E27F1B1FFB5C4C758178305D48E4E78C021E4299BD88700CAD19DAB92C534B4CFF7554");
        byte[] cipher = Hex.hexToByteArray("38615d1310d2e327513b8b6ee483bfaed593d867f65e3150101aa844e6815a9a67af983064157721f64c89db098ffef4496213fe267db8fc0994c05e143a732a");
        byte[] encData = CipherSelfTestCase.symmetric(msg, "ARIA/CBC/NoPadding", "ARIA", key, iv, 1);
        byte[] decData = CipherSelfTestCase.symmetric(encData, "ARIA/CBC/NoPadding", "ARIA", key, iv, 2);
        if (!Arrays.equals(cipher, encData) || !Arrays.equals(msg, decData)) {
            throw new KSXRuntimeException("testARIA256CBC SelfTest Failed");
        }
    }

    void testARIA256CFB8() throws KSXRuntimeException {
        byte[] key = Hex.hexToByteArray("24DBF1EF6527206248043566B68B7089165594D163419AA28171A7E1D1963C5A");
        byte[] iv = Hex.hexToByteArray("036EAC5F2B8BD4B2175506A6E0524680");
        byte[] msg = Hex.hexToByteArray("C0A5B92323D55AED01D0EFDC6A303CF563BC76E6F032FAEA72443FED33E009F140B4E0B077F2C182E05013824154DCECECCE35CADCA6FCD000D7C884D7BC22A23D8BF2BE851F372BC7FAFB1C7B8A8709");
        byte[] cipher = Hex.hexToByteArray("5047325d871ba53d442b2e318a3eb127216268b28494d8975d460426a57183a99fcc2b6254cdb8adee8460e3c04c05635f3be3598facbae6af7f9862b42b47fcb17f2e2a480d47e0d5d4df1b77b73d7e");
        byte[] encData = CipherSelfTestCase.symmetric(msg, "ARIA/CFB8/NoPadding", "ARIA", key, iv, 1);
        byte[] decData = CipherSelfTestCase.symmetric(encData, "ARIA/CFB8/NoPadding", "ARIA", key, iv, 2);
        if (!Arrays.equals(cipher, encData) || !Arrays.equals(msg, decData)) {
            throw new KSXRuntimeException("testARIA256CFB8 SelfTest Failed");
        }
    }

    void testARIA256CFB128() throws KSXRuntimeException {
        byte[] key = Hex.hexToByteArray("47ABD354166E402D6A1889B3F36EF75B6030D59675E424AE5ED490F1435AA212");
        byte[] iv = Hex.hexToByteArray("1C5B36CCC71230C8570F6BC1B07FF2E6");
        byte[] msg = Hex.hexToByteArray("51AD738070C9F0D44CE9B84BBF4D96D012E7487A161A326B70C354F835767A0AB70AB2C39E5703653A3E5C7017EACB0BD256EFA52C0FB0DC5D3C2D9EABDBF68E3A4D7CA92615C629CFDD65A974B9A98B04B01597307911E4C582DFFC3937D03B");
        byte[] cipher = Hex.hexToByteArray("607ea2ffc20b7b8b09fe067d9788f4e14f9228382ee2339f976e867494ae026498cfffde420aeb318010c581a8069a720417a517bd21767867085b0bd6f415053ed42ef57dda4f2b89986601cf94620b0811282379bb46e5bd419d20f1bd58bd");
        byte[] encData = CipherSelfTestCase.symmetric(msg, "ARIA/CFB128/NoPadding", "ARIA", key, iv, 1);
        byte[] decData = CipherSelfTestCase.symmetric(encData, "ARIA/CFB128/NoPadding", "ARIA", key, iv, 2);
        if (!Arrays.equals(cipher, encData) || !Arrays.equals(msg, decData)) {
            throw new KSXRuntimeException("testARIA256CFB128 SelfTest Failed");
        }
    }

    void testARIA256OFB() throws KSXRuntimeException {
        byte[] key = Hex.hexToByteArray("8380B878308B9FE8B4CDB93EFE4399174FFEA1964BDEBD4B519E515A08106FD9");
        byte[] iv = Hex.hexToByteArray("3AAA4E79E642F76990164277DC0F0179");
        byte[] msg = Hex.hexToByteArray("5B477BEBA5C81CD9A8976A003FF13B3007D425F46EC137750CC1E69C37A64A77D1938BDE65BE965672751435075F9C0790052731F090C5D4D0D48FF4368EDDDA59EAB9B6B44993895940364288E4FA2780453B7795380B4E835A24C8025120699A55EDBDB9F07B3B0301B76FE907BD59");
        byte[] cipher = Hex.hexToByteArray("c71349c1ec7678152b421f6b94afed43d5a10be876e55450d3d845c8de4b1fe54d270235165a37f7d84062c68a000bb7ef67163ba210835dc24e0882bedef3eafa8b7d5ad71a8be9118067f0423079a7eac17a1acd03754ebb3d88dcc1261abbafb49d1a04f896f247b2eed588f04215");
        byte[] encData = CipherSelfTestCase.symmetric(msg, "ARIA/OFB/NoPadding", "ARIA", key, iv, 1);
        byte[] decData = CipherSelfTestCase.symmetric(encData, "ARIA/OFB/NoPadding", "ARIA", key, iv, 2);
        if (!Arrays.equals(cipher, encData) || !Arrays.equals(msg, decData)) {
            throw new KSXRuntimeException("testARIA256OFB SelfTest Failed");
        }
    }

    void testARIA256CTR() throws KSXRuntimeException {
        byte[] key = Hex.hexToByteArray("7D9C491084406FAACD588D60C2767DEE3DDB0F399D3BB63418C0810453514D63");
        byte[] iv = Hex.hexToByteArray("2F133A42E6305BB258D8B2049F865B2F");
        byte[] msg = Hex.hexToByteArray("E985077387B2AC3C41837B49ED48130D40B1F456E391362CC9E17BFCC10822F6475ABDB3A0DEC51B26548D86E0767521557F6094A2EA68E1CB7CD08F4F843909FF6219410F466A986E3A9F025363DC65198366434CC3599904B0990771830A30B8A50464FD72017DC13E99066F95B1A331C7EFAC08A4711C1C85BEAA518CFC35");
        byte[] cipher = Hex.hexToByteArray("1CF8AEE21ADB35B3FE5634F15C5F1C4DC58541849E1173EDB9761F47A9E6A57B2C6579BC309352AD9B66904CB6C251BFFA23138F4AE16175E2C16130201E657E8E22BA3A4E9B8CCDB82A33001D931BEEE02CF39831F1C67C86F5E95D88D3E7D290EE6CFFCBA349AD5E5E2DC94B6C13E32F6BB93DF996065E5C10F2301559D9D9");
        byte[] encData = CipherSelfTestCase.symmetric(msg, "ARIA/CTR/NoPadding", "ARIA", key, iv, 1);
        byte[] decData = CipherSelfTestCase.symmetric(encData, "ARIA/CTR/NoPadding", "ARIA", key, iv, 2);
        if (!Arrays.equals(cipher, encData) || !Arrays.equals(msg, decData)) {
            throw new KSXRuntimeException("testARIA256CTR SelfTest Failed");
        }
    }

    void testSEEDECB() throws KSXRuntimeException {
        byte[] key = Hex.hexToByteArray("FB1B29A375AEA4D0551C754B57F96FED");
        byte[] iv = Hex.hexToByteArray("008C49DFEAD697B75DCFF9E31EF3934B");
        byte[] msg = Hex.hexToByteArray("8CFDBD32E8F3FB5DF5622F78C1EB0DB5C2FED0B624E62BAD620683C214625BB5");
        byte[] cipher = Hex.hexToByteArray("d1abc1da5d83024d04d201ab0503a89837a8ffafc816f7f413d1c41ac9e0add9");
        byte[] encData = CipherSelfTestCase.symmetric(msg, "SEED/ECB/NoPadding", "SEED", key, iv, 1);
        byte[] decData = CipherSelfTestCase.symmetric(encData, "SEED/ECB/NoPadding", "SEED", key, iv, 2);
        if (!Arrays.equals(cipher, encData) || !Arrays.equals(msg, decData)) {
            throw new KSXRuntimeException("testSEEDECB SelfTest Failed");
        }
    }

    void testSEEDCBC() throws KSXRuntimeException {
        byte[] key = Hex.hexToByteArray("086110C493D1C6CF292BA2BA2C183A13");
        byte[] iv = Hex.hexToByteArray("023549F46A13A82B108379995F10A5D8");
        byte[] msg = Hex.hexToByteArray("96CC8A211E50ED6C1BFE36D84189DC4EE8B81E616366F4798ECE4531A7055AFD5FC893103078587DF063539CA745DCAF");
        byte[] cipher = Hex.hexToByteArray("a3cd4334b40b965998b0da9af1febf4190b22e49dd7b93736e1d320f39a0e2ee8695277f664df5acecbbbbfcc6acff7d");
        byte[] encData = CipherSelfTestCase.symmetric(msg, "SEED/CBC/NoPadding", "SEED", key, iv, 1);
        byte[] decData = CipherSelfTestCase.symmetric(encData, "SEED/CBC/NoPadding", "SEED", key, iv, 2);
        if (!Arrays.equals(cipher, encData) || !Arrays.equals(msg, decData)) {
            throw new KSXRuntimeException("testSEEDCBC SelfTest Failed");
        }
    }

    void testSEEDCFB8() throws KSXRuntimeException {
        byte[] key = Hex.hexToByteArray("9E0FB5C5B8E6F8DF056F4D52954A606C");
        byte[] iv = Hex.hexToByteArray("8ADC935A7152F049D3E261CC0654237E");
        byte[] msg = Hex.hexToByteArray("49C278E8119C9CA59DEEFBC3D0E6A16D4091F2C78BE59B1AEA04C83108BF98031459CD92148EC9127DD4B64E01E10448A96D181F233942365D50F093538D2285");
        byte[] cipher = Hex.hexToByteArray("7110a2ad8f84e6546ae8fde394c4d324732114f51d89fbc99b84875e0d9578d603ce218f802fbcb1e128f20547949e703cc0cc6ad344878bf19ca714f49d1d60");
        byte[] encData = CipherSelfTestCase.symmetric(msg, "SEED/CFB8/NoPadding", "SEED", key, iv, 1);
        byte[] decData = CipherSelfTestCase.symmetric(encData, "SEED/CFB8/NoPadding", "SEED", key, iv, 2);
        if (!Arrays.equals(cipher, encData) || !Arrays.equals(msg, decData)) {
            throw new KSXRuntimeException("testSEEDCFB SelfTest Failed");
        }
    }

    void testSEEDCFB128() throws KSXRuntimeException {
        byte[] key = Hex.hexToByteArray("255D1E8A6CC6656FCEA9E5B9D0447044");
        byte[] iv = Hex.hexToByteArray("ECFA6D2AE457CEE6554F42BA8FC6AB4E");
        byte[] msg = Hex.hexToByteArray("A456D299BF4C5904B7F3F2CDE416CFAA32C05AB073462572FA87246D65731AA4BACA5187B6268E19613C4452E65F08C2A24645787B0E31227382FF767C9B57D08F43031CF85EEBF6F40B43107D2703D4");
        byte[] cipher = Hex.hexToByteArray("22af28f34b844c13b92891ec7b94ec2e4045fd0c71d68dd670819586c57a1859355cf8d73e9fbb1c3f1607112740fac1ff3a60770e2dcbb700f9d942645481528e6c9884868e81be4fc05a74d629a45b");
        byte[] encData = CipherSelfTestCase.symmetric(msg, "SEED/CFB128/NoPadding", "SEED", key, iv, 1);
        byte[] decData = CipherSelfTestCase.symmetric(encData, "SEED/CFB128/NoPadding", "SEED", key, iv, 2);
        if (!Arrays.equals(cipher, encData) || !Arrays.equals(msg, decData)) {
            throw new KSXRuntimeException("testSEEDCFB SelfTest Failed");
        }
    }

    void testSEEDOFB() throws KSXRuntimeException {
        byte[] key = Hex.hexToByteArray("6514974BA2B8D83EE9C83C9B7E464919");
        byte[] iv = Hex.hexToByteArray("4A48501F2EFC57E397E957CF5277A828");
        byte[] msg = Hex.hexToByteArray("A2B2B9F1904B040D83DF41A5107CDAC91362A05AFF06BC27725DE8560D56DF0682A91132EECF9CD86A51785BDC46F12714175B931486010BAEEF5E6E55CD90B52D7F09D6644D88E7C5A647878AAB767A72F2E99314830ED7732721DFD2E3A37E");
        byte[] cipher = Hex.hexToByteArray("b7acf4a9110513e00b6b3bcb3910c3c8ff64c1c2891ba5fdd093684563eaa48bd56f01adf8602bbfdb6fb820e016359b1c6d6df9205c740cbe32d13271ee9bb57741077b09895d41ad706e658d80da74570bb6e94f9bef41f2070c185245cf2e");
        byte[] encData = CipherSelfTestCase.symmetric(msg, "SEED/OFB/NoPadding", "SEED", key, iv, 1);
        byte[] decData = CipherSelfTestCase.symmetric(encData, "SEED/OFB/NoPadding", "SEED", key, iv, 2);
        if (!Arrays.equals(cipher, encData) || !Arrays.equals(msg, decData)) {
            throw new KSXRuntimeException("testSEEDOFB SelfTest Failed");
        }
    }

    void testSEEDCTR() throws KSXRuntimeException {
        byte[] key = Hex.hexToByteArray("CABF08A499CBB082BD6418EFC2B4520B");
        byte[] iv = Hex.hexToByteArray("5878B222A806CBD3E88E99712DA000A9");
        byte[] msg = Hex.hexToByteArray("81EE75653654FCF17A15525C2B686B22EC331F077816204636AB2FEB0E0C8F7E7C96BAE1E2ED547B23405D976DCFAA0656AA960C2BBBF57A840649171C303944E03E3EE046A0A06AE06EA16531F1F701BF6580F769FD32B5FB2851BB0013E445D86F692A0A74C804DB2686911FD73B59");
        byte[] cipher = Hex.hexToByteArray("e68ae41fadb6021bbabbc51de7e26f8ffe323757c7d8cd5841d8549df0e1f0d5fc1429541b1e40dda31f22ef8a5cfac129f09252f32ac651d741b000c32aa64900f6996571c4316db0f947be385b67ea408748f04c997da0db71b05f43aeadfa59242814113e0b623543a7c5d8781b32");
        byte[] encData = CipherSelfTestCase.symmetric(msg, "SEED/CTR/NoPadding", "SEED", key, iv, 1);
        byte[] decData = CipherSelfTestCase.symmetric(encData, "SEED/CTR/NoPadding", "SEED", key, iv, 2);
        if (!Arrays.equals(cipher, encData) || !Arrays.equals(msg, decData)) {
            throw new KSXRuntimeException("testSEEDCTR SelfTest Failed");
        }
    }

    void testRSAES224ENT() throws NoSuchAlgorithmException, NoSuchProviderException, InvalidKeySpecException, NoSuchPaddingException, InvalidKeyException, InvalidAlgorithmParameterException, BadPaddingException, IllegalBlockSizeException, IllegalStateException {
        BigInteger n = new BigInteger(1, Hex.hexToByteArray("EF1CF3EDC3EEFE1B11F66AEE3B7C0B38779929E19062980DA40B2F65C561A5311508BD2596CA68702D1A2DD8F0A27CF4AC7A533795A6B177FCFD534FF104337B1B231CBC733ADFD54F9A2B4EDB2FCC9A2D624C719D5469FBE71C966C7DB930BF4E5714DCF995D940F3F96DA011C523709D5F5E68532AA1E37D9C4A23182D9911FBF81CCC4B18B7BE04C14CAA91D87A4DB890B3B43B404E0B8BAE1FCE7F6106749B8D56DE122BCD609A3A887365B213555ED14CDE0FE0227EC898BBA43F9BF44EAD7316BF6E06F45F9F261D76ECB7472E8B093DC9F1FAB8A74774E78EC26793242CD6F856820E17DFEBC89F3EB497FBED180BCCEE38A6F9AC3EFF0EBA52F3C187"));
        BigInteger e = new BigInteger(1, Hex.hexToByteArray("010001"));
        byte[] msg = Hex.hexToByteArray("6DFBE6A88CC8A01B3ECC9E154940F87AFCC89D737C9C6D40E841BAD9");
        byte[] cipher = Hex.hexToByteArray("AD0342F8A5F060D984C68949A29855431B0874A4F8E154ED827D640CDA7BF6CB410D6DC241122618AFBA3085A9C47D308631225588AA663155BEE492F656C816BE3C7F88D32F53753EB005493FA66A964EC28F15ABCA1E15BE7BB242DE5DE19C33E24F4F37EA85B4E09A204E9DBDF6054895D99C022FC6027CB3077EB2C083058FB8CEC0AC548FE1E8B407890C693BD97E1367FBF46F5C9F99E87853487BF91C1F35F96D7404082085767E94A8754FC9A8D1D7156045A6B931C79D0D4AFCCC7C72C3C7D460EC59F603F16823312A5A862D87EA42BF6015D87B23DCF4ED0813CE4B16D6FCF757D617DF8DAEFC81C4BBA8F7CC2B6AB6FC25DFEB3723AD407119A1");
        KeyFactory kf = KeyFactory.getInstance("RSA", InitechProvider.NAME);
        RSAPublicKeySpec publicKeySpec = new RSAPublicKeySpec(n, e);
        PublicKey pubilcKey = kf.generatePublic(publicKeySpec);
        Cipher rsa = Cipher.getInstance("RSA/ECB/OAEPPadding", InitechProvider.NAME);
        byte[] seed = new byte[28];
        rsa.init(1, (Key)pubilcKey, new RSAOAEPParameterSpec("SHA224", seed));
        byte[] encData = rsa.doFinal(msg);
        if (!Arrays.equals(cipher, encData)) {
            throw new KSXRuntimeException("testRSAES224ENT SelfTest Failed");
        }
    }

    void testRSAES224DET() throws NoSuchAlgorithmException, NoSuchProviderException, InvalidKeySpecException, NoSuchPaddingException, InvalidKeyException, InvalidAlgorithmParameterException, BadPaddingException, IllegalBlockSizeException, IllegalStateException {
        BigInteger n = new BigInteger(1, Hex.hexToByteArray("DD44EF59B5401C1F0CCE2EA18E2F43E7A7107CF822F5BDE3E9BA69A1813A75DDCC6FB0038B66148A0A1871B35069E4E993C441A475AF41695CDD3C633FA325984EE28F8D92F0A4094B27644285E26CF8A63BB5BA56E39360EE595E67BFD8CE6258257D3BF245EB2F108BA4F3B116F9A7ADD2B604EECB011E8EDB08FE7D2F53657533017E59F51D833466682F43BC9CD55C39B1E4BA56A549F87624BC38285A5131D12E8B48C8A7DC14A921DC7387BAE95AA5659208FD419B327345DBD7F164D49BBC72C348A373F9793EF953DAFC2BD72E65264A969382DAE8B0883F5166149E1C364F0EF7886460F6BBB9C280F126C2A34A24A1FF567EB3B4CA8BE5F51D3F7F"));
        BigInteger d = new BigInteger(1, Hex.hexToByteArray("D3E9E789F50AF763CD71D23B2BDF9C9B49E782A86B630B8032233473573ED40F9E72011303F8BA07127022C4E768E3B1C8B1976498C9464E50801A0BBBB0D69AAE4251D5F73ABCFFA198926F1BA6A552E6721D33DF6F8980B605A5629BDC5A9CD96ADE2C0176712DDC1CC1D81A6CB4A1A0DA43352A3606566BDEDF184C707CDED099B501E0F6EA5E2C1F07B9CFDF81199EE1DD5F927C5C39E274A0076B560AE52E5C8EF22B538C68FED69AB6209F171D9D99B1738B40BA7436D26518FF9A06CB73FC85C6513EB41EB16B04DA3D8EC3E9A743147C37EE8036EB1B39D682888821D9E1178EA653CFE107EAFA93BAF00ADCEF39F3730605D3F28CF4F8560F0A1F51"));
        BigInteger e = new BigInteger(1, Hex.hexToByteArray("010001"));
        byte[] msg = Hex.hexToByteArray("690F93D87B02539DB2113FE5B38F2767680FCAF1B67847488401B075");
        byte[] cipher = Hex.hexToByteArray("A4D24AF7AAABE1ECE7CD2FF7E4BF729AB0941612D664C597FE9E40899977399EE1472BF2963E421A2FFE4AB760CBC202FED94FDF444334B96D042846012F8BBFE1BBF6767583C8221F13E5B87582B501D6B689431D205E0221FE20CB0274BBDE57B386EAF89A9106EF30540F0ADA1878ED9BD73D28853E9697E9886A4ACE6723EF48A4BA834E58F56C5C4C13A4BEE628B617E23892250498852658B8761DF47FBD37150B15293CD99DE84725C711753883298C08737C42CFAC98A0BB742F81672751163D131EF6837898F4CE28BA5D0B86A96796B6074F536A73961DC10D64CB0F9DD968BC639825C6F7BD366F39997BCA56F5483052251EBD5D26DA4677C429");
        KeyFactory kf = KeyFactory.getInstance("RSA", InitechProvider.NAME);
        RSAPrivateKeySpec privateKeySpec = new RSAPrivateKeySpec(n, d);
        PrivateKey privateKey = kf.generatePrivate(privateKeySpec);
        Cipher rsa = Cipher.getInstance("RSA/ECB/OAEPPadding", InitechProvider.NAME);
        byte[] seed = new byte[28];
        rsa.init(2, (Key)privateKey, new RSAOAEPParameterSpec("SHA224", seed));
        byte[] decData = rsa.doFinal(cipher);
        if (!Arrays.equals(msg, decData)) {
            throw new KSXRuntimeException("testRSAES224DET SelfTest Failed");
        }
    }

    void testRSAES256ENT() throws NoSuchAlgorithmException, NoSuchProviderException, InvalidKeySpecException, NoSuchPaddingException, InvalidKeyException, InvalidAlgorithmParameterException, BadPaddingException, IllegalBlockSizeException, IllegalStateException {
        BigInteger n = new BigInteger(1, Hex.hexToByteArray("A650812736D695D5543BEC83252F3DB39341E1ABD708B66C7047D8986CFB4D618F25521EFBD6DE62F5862A222FC63C60061AC80DFE1A5D58BC1A025806C4BC2DE5BF8A500F54930061BD91DD911FBFCCAA917FC9E433BD1E87D06626FE64F113B29FD1607E8EC812C83883D460517D9423CA37801159AC72A99B704639F58A48423EEBE317FD497695310CD5F4C435012854F441F8A856BD65A12360B087CA4524DFBEE377B981EBBB84EA7B0AC47AC9255898281AD0FF114FA4297F77E5FF0E2AC67FA28F1F7AA8363D4A1B6B408F1F12C1FAA5C26ABBCDF8267389E922033141EC3BD4B9454ACC6AEE7F3729F96CF2A844B30D5ED4D0BFDC2D715E4B150CE5"));
        BigInteger e = new BigInteger(1, Hex.hexToByteArray("010001"));
        byte[] msg = Hex.hexToByteArray("AEEA9B364A17EE13EB45F213B4D99E54AD06DEFA418B08E0CA1F22E6");
        byte[] cipher = Hex.hexToByteArray("2D3982E562FDB90D4E6C6BD24C3980BF669498D6FE16AC53C25C5264BEB36F0B167A826DC254A5C0B92F231D307E9C18B997CC5769573AE2EAF627E5477105B268DAC82D527A5552C203551520AAFCFCFF04D9582B4659E29973F30425065224E37216BB6351B05B29A06DA5FE9603BFD25BEFA47E596731E77F6F1F424BF844510BC9A18D99DA268CF7BF0015AA2395A0A82AEB06039317C04222E4A69B7469BD6667018D4CFB17ADF49489F026E224FFC407F13FB0FCDEC67BB7D57F8D60F9F9442F320DDEC7991B9360F03F8A846D760519CAC0318FA4F69EFC422A4F785BFAD8854CBA8E1917E1692B35DBEC2ECCEE9D8BC53B53DB6FD2C1C16DA15CB98B");
        KeyFactory kf = KeyFactory.getInstance("RSA", InitechProvider.NAME);
        RSAPublicKeySpec publicKeySpec = new RSAPublicKeySpec(n, e);
        PublicKey pubilcKey = kf.generatePublic(publicKeySpec);
        Cipher rsa = Cipher.getInstance("RSA/ECB/OAEPPadding", InitechProvider.NAME);
        byte[] seed = new byte[32];
        rsa.init(1, (Key)pubilcKey, new RSAOAEPParameterSpec("SHA256", seed));
        byte[] encData = rsa.doFinal(msg);
        if (!Arrays.equals(cipher, encData)) {
            throw new KSXRuntimeException("testRSAES256ENT SelfTest Failed");
        }
    }

    void testRSAES256DET() throws NoSuchAlgorithmException, NoSuchProviderException, InvalidKeySpecException, NoSuchPaddingException, InvalidKeyException, InvalidAlgorithmParameterException, BadPaddingException, IllegalBlockSizeException, IllegalStateException {
        BigInteger n = new BigInteger(1, Hex.hexToByteArray("DD87C09D2B749DA093A73A5C04205917C9886C9ED9942A8E7F2FBC769A6B37F405D56D85F10F9C8AB81528824EFB0571C78C1BA0F3794C497182DEF34447C60F8514B986E61BFC4F67307F5AA5245C5253065C15A962BAFE846FECB742D264983F392F386943CB4191EC466963233D343CD2785CAF27BF2359C32D2C4F08DE24AF26D9811D172752BC6F4C558700D61F0179807CDA97C6C1D2468229D68A4516E93CD47CD32F57493E3D3BC5933051B4CE127F69F542E70CAE1CB835892274061F1C3CC2B765D72BB2DCEA0F8E79865D0FD767AEDB06A68C1646D26CC428882AE9F7A20BD8B77CA2A04A85D49A4B0B4F9B1159382A0D397A480F0AF5A48DEBA3"));
        BigInteger d = new BigInteger(1, Hex.hexToByteArray("99488C7374151DD875325A043F6AAE9AF1FCEEC3BF63714AE6207EB44646F8116653F6939E6821F62E479D9F04CD5648133E1403FB542C000985F784B5F33FCD8F54F6C6ABC4238241B161FDB39504388670391957FA13CF768E076A0AD78F79750DB969BC1487B89822B7D9C61A250857297ED50AF162A5304007AAC80927C34ABD807D282F1EC27AD57E82FD36E60A78B4474D458B3A76B9496360528FC7BC0A217CB85A6AF2AA1268CCA9E51739CF5ABE020D75F1C14976FF5BF84D503785535E0FBC4FBDBBBAD2CBF36E6668D4E1AEE9CFC59D509681903C180F27D0D1A2803E56AF0BD21DE34BC3E10C410EE84C0AC91A0DA090A13C8A486BFEE6FFDFE1"));
        BigInteger e = new BigInteger(1, Hex.hexToByteArray("010001"));
        byte[] msg = Hex.hexToByteArray("ABE9756AF11D0F3EEFB982E840BAF6305FBDE23DB57DC11E16EFD90C");
        byte[] cipher = Hex.hexToByteArray("D6594289B5D7FB6229ED0046FECC13CCEFB5361ED2C6C30F823554FC3D7AAE4583C78AAD772AFB9BB7D85EA0E72A826614B3A6657741048C91320C3EE471102B59FDC86267DE7CB03CFD20F537FEC626CB201BFCBDCC77E639E0B932B3BBD41B9A76D030C978B924120AA7452FF3C68F717EFFC9D585C432BA595916B5D3A2952595BF859C71F63C709F13600F2F1EBE9691838FBDDB74332683280AF3FDF709FC9E27F32833C9C8E772F63BA6B8296F70FEB22BFDAE4DB905858980513D6479B5132C8000C962B33FF6F3FB6C6953A5C46A5DA4AB1AAAF93144FB22E744737910BB42F5F2024A72A2CC3EEEE9605C5FEE7D2EA81C4667D639189846387C8A11");
        KeyFactory kf = KeyFactory.getInstance("RSA", InitechProvider.NAME);
        RSAPrivateKeySpec privateKeySpec = new RSAPrivateKeySpec(n, d);
        PrivateKey privateKey = kf.generatePrivate(privateKeySpec);
        Cipher rsa = Cipher.getInstance("RSA/ECB/OAEPPadding", InitechProvider.NAME);
        byte[] seed = new byte[32];
        rsa.init(2, (Key)privateKey, new RSAOAEPParameterSpec("SHA256", seed));
        byte[] decData = rsa.doFinal(cipher);
        if (!Arrays.equals(msg, decData)) {
            throw new KSXRuntimeException("testRSAES256DET SelfTest Failed");
        }
    }

    void testARIA128AllMode() {
        this.testARIA128ECB();
        this.testARIA128CBC();
        this.testARIA128CFB128();
        this.testARIA128OFB();
        this.testARIA128CTR();
    }

    void testARIA192AllMode() {
        this.testARIA192ECB();
        this.testARIA192CBC();
        this.testARIA192CFB128();
        this.testARIA192OFB();
        this.testARIA192CTR();
    }

    void testARIA256AllMode() {
        this.testARIA256ECB();
        this.testARIA256CBC();
        this.testARIA256CFB128();
        this.testARIA256OFB();
        this.testARIA256CTR();
    }

    void testSEEDAllMode() {
        this.testSEEDECB();
        this.testSEEDCBC();
        this.testSEEDCFB128();
        this.testSEEDOFB();
        this.testSEEDCTR();
    }

    void testRSAESAllMode() {
        try {
            this.testRSAES224ENT();
            this.testRSAES224DET();
            this.testRSAES256ENT();
            this.testRSAES256DET();
        }
        catch (NoSuchPaddingException e) {
            e.printStackTrace();
        }
        catch (InvalidKeyException e) {
            e.printStackTrace();
        }
        catch (BadPaddingException e) {
            e.printStackTrace();
        }
        catch (IllegalBlockSizeException e) {
            e.printStackTrace();
        }
        catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        }
        catch (NoSuchProviderException e) {
            e.printStackTrace();
        }
        catch (InvalidKeySpecException e) {
            e.printStackTrace();
        }
        catch (InvalidAlgorithmParameterException e) {
            e.printStackTrace();
        }
        catch (IllegalStateException e) {
            e.printStackTrace();
        }
    }

    private static byte[] symmetric(byte[] msg, String symAlg, String keyAlg, byte[] key, byte[] iv, int mode) {
        byte[] cipherData = null;
        SecretKey sk = null;
        SecretKeyFactory secretFactory = null;
        try {
            secretFactory = SecretKeyFactory.getInstance(keyAlg, InitechProvider.NAME);
            SecretKeySpec scretSpec = new SecretKeySpec(key, keyAlg);
            sk = secretFactory.generateSecret(scretSpec);
            Cipher cipher = Cipher.getInstance(symAlg, InitechProvider.NAME);
            cipher.init(mode, (Key)sk, new IvParameterSpec(iv));
            cipherData = cipher.doFinal(msg);
        }
        catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
        }
        catch (NoSuchProviderException e) {
            e.printStackTrace();
        }
        catch (InvalidKeySpecException e) {
            e.printStackTrace();
        }
        catch (InvalidKeyException e) {
            e.printStackTrace();
        }
        catch (InvalidAlgorithmParameterException e) {
            e.printStackTrace();
        }
        catch (NoSuchPaddingException e) {
            e.printStackTrace();
        }
        catch (BadPaddingException e) {
            e.printStackTrace();
        }
        catch (IllegalBlockSizeException e) {
            e.printStackTrace();
        }
        catch (IllegalStateException e) {
            e.printStackTrace();
        }
        return cipherData;
    }

    public void selfTest() {
        this.testARIA128AllMode();
        this.testARIA192AllMode();
        this.testARIA256AllMode();
        this.testSEEDAllMode();
        this.testRSAESAllMode();
        this.testSEED();
        this.testARIA128();
        this.testARIA192();
        this.testARIA256();
        this.testRSAES();
    }
}

